package main

import (
	"io"
	"bytes"
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"strconv"
	"time"
	"math"
)

type TokenHolder struct {
	Address string
	Balance float64
}


// FetchTokenSupply retrieves the total supply and decimals of a token using `getTokenSupply`.
func FetchTokenSupply(tokenAddress string) (float64, int, error) {
	payload := map[string]interface{}{
		"jsonrpc": "2.0",
		"id":      1,
		"method":  "getTokenSupply",
		"params":  []interface{}{tokenAddress},
	}

	body, err := json.Marshal(payload)
	if err != nil {
		return 0, 0, fmt.Errorf("error marshalling request payload: %w", err)
	}

	for _, rpcURL := range solanaRPCURLs {
		resp, err := http.Post(rpcURL, "application/json", bytes.NewBuffer(body))
		if err != nil {
			log.Printf("Error querying Solana RPC (%s): %v", rpcURL, err)
			continue
		}
		defer resp.Body.Close()

		var response map[string]interface{}
		if err := json.NewDecoder(resp.Body).Decode(&response); err != nil {
			log.Printf("Error decoding response from %s: %v", rpcURL, err)
			continue
		}

		log.Printf("getTokenSupply response from %s: %+v", rpcURL, response)

		if result, ok := response["result"].(map[string]interface{}); ok {
			if value, ok := result["value"].(map[string]interface{}); ok {
				amountStr, ok := value["amount"].(string)
				if !ok {
					continue
				}
				amount, err := strconv.ParseFloat(amountStr, 64)
				if err != nil {
					continue
				}

				decimalsFloat, ok := value["decimals"].(float64)
				if !ok {
					continue
				}
				decimals := int(decimalsFloat)

				return amount, decimals, nil
			}
		}
	}

	return 0, 0, fmt.Errorf("all RPC queries failed or returned unexpected response")
}

// FetchLargestTokenAccounts retrieves the largest holders of a token using `getTokenLargestAccounts`.
func FetchLargestTokenAccounts(tokenAddress string) ([]TokenHolder, error) {
	payload := map[string]interface{}{
		"jsonrpc": "2.0",
		"id":      1,
		"method":  "getTokenLargestAccounts",
		"params":  []interface{}{tokenAddress},
	}

	body, err := json.Marshal(payload)
	if err != nil {
		return nil, fmt.Errorf("error marshalling request payload: %w", err)
	}

	retries := 3
	delay := time.Second * 2
	for _, rpcURL := range solanaRPCURLs {
		for attempt := 1; attempt <= retries; attempt++ {
			resp, err := http.Post(rpcURL, "application/json", bytes.NewBuffer(body))
			if err != nil {
				log.Printf("Error querying Solana RPC (%s): %v", rpcURL, err)
				if attempt < retries {
					time.Sleep(delay)
					continue
				}
				break
			}
			defer resp.Body.Close()

			var response map[string]interface{}
			if err := json.NewDecoder(resp.Body).Decode(&response); err != nil {
				log.Printf("Error decoding response from %s: %v", rpcURL, err)
				break
			}

			log.Printf("getTokenLargestAccounts response from %s: %+v", rpcURL, response)

			holders := []TokenHolder{}
			if result, ok := response["result"].(map[string]interface{}); ok {
				if accounts, ok := result["value"].([]interface{}); ok {
					for _, account := range accounts {
						acc := account.(map[string]interface{})
						balance, ok := acc["uiAmount"].(float64)
						if !ok {
							log.Printf("Invalid balance format in account: %+v", acc)
							continue
						}
						address, ok := acc["address"].(string)
						if !ok {
							log.Printf("Invalid address format in account: %+v", acc)
							continue
						}
						holders = append(holders, TokenHolder{
							Address: address,
							Balance: balance,
						})
					}
					return holders, nil
				} else {
					log.Printf("Unexpected 'value' field format in response: %+v", result["value"])
				}
			} else {
				log.Printf("Unexpected 'result' field format in response: %+v", response["result"])
			}
		}
	}

	return nil, fmt.Errorf("all RPC queries failed or returned unexpected response")
}

// AnalyzeWallets checks top holders for signs of bundled wallets.
func AnalyzeWallets(holders []TokenHolder) string {
	for _, holder := range holders {
		if holder.Balance > 0.3 { // Example threshold for bundled wallets
			return "❌ Bundled Detected"
		}
	}
	return "✅ Not Bundled"
}

// AssessRugPullRisk provides a basic risk assessment based on top holder concentration.
func AssessRugPullRisk(holders []TokenHolder) string {
	if len(holders) > 0 && holders[0].Balance > 0.5 { // Example threshold for risk
		return "High Risk (Top Holder Dominance)"
	}
	return "Low Risk"
}



func FormatMarketCap(supply float64, decimals int, price float64) string {
    // Adjust the supply based on decimals
    adjustedSupply := supply / math.Pow10(decimals)
    marketCap := adjustedSupply * price

    // Format the market cap for readability (K, M, B)
    switch {
    case marketCap >= 1_000_000_000:
        return fmt.Sprintf("%.2fB", marketCap/1_000_000_000)
    case marketCap >= 1_000_000:
        return fmt.Sprintf("%.2fM", marketCap/1_000_000)
    case marketCap >= 1_000:
        return fmt.Sprintf("%.2fK", marketCap/1_000)
    default:
        return fmt.Sprintf("%.2f", marketCap)
    }
}





// FetchTokenPriceFromMoralis retrieves token price from Moralis API.
func FetchTokenPriceFromMoralis(tokenAddress string) (float64, error) {
    url := fmt.Sprintf("https://solana-gateway.moralis.io/token/mainnet/%s/price", tokenAddress)

    log.Printf("Moralis API Request URL: %s", url) // Log the request URL

    req, err := http.NewRequest("GET", url, nil)
    if err != nil {
        return 0, fmt.Errorf("error creating request: %w", err)
    }

    // Set required headers
    req.Header.Add("Accept", "application/json")
    req.Header.Add("X-API-Key", "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJub25jZSI6IjI1ZmNhMTA3LTE2YjQtNGRmZS04NGVkLWRhNTk0MzIxMDMyOCIsIm9yZ0lkIjoiNDI0MjMyIiwidXNlcklkIjoiNDM2MzA2IiwidHlwZUlkIjoiOWMwODI3MDctOGMzMy00NjA4LWEwMGMtNDJhMjM1Yzg4YTlhIiwidHlwZSI6IlBST0pFQ1QiLCJpYXQiOjE3MzYxMTQ3NzQsImV4cCI6NDg5MTg3NDc3NH0.Uqvk_MhsIcU921pNzrIQp-n5A4Qk-Ywl6Tm8_IzF4o0") // Replace with your actual API key

    log.Printf("Moralis API Headers: %v", req.Header) // Log headers

    res, err := http.DefaultClient.Do(req)
    if err != nil {
        return 0, fmt.Errorf("error calling Moralis API: %w", err)
    }
    defer res.Body.Close()

    // Log the HTTP status code
    log.Printf("Moralis API status code: %d", res.StatusCode)

    if res.StatusCode != http.StatusOK {
        body, _ := io.ReadAll(res.Body)
        log.Printf("Moralis API error response: %s", string(body))
        return 0, fmt.Errorf("unexpected status code from Moralis API: %d", res.StatusCode)
    }

    body, err := io.ReadAll(res.Body)
    if err != nil {
        return 0, fmt.Errorf("error reading response body: %w", err)
    }

    // Log the raw response body for debugging
    log.Printf("Raw Moralis API response: %s", string(body))

    var result map[string]interface{}
    if err := json.Unmarshal(body, &result); err != nil {
        return 0, fmt.Errorf("error decoding Moralis API response: %w", err)
    }

    // Log the parsed JSON for debugging
    log.Printf("Decoded Moralis API response: %v", result)

    // Extract and return the usdPrice field
    if usdPrice, ok := result["usdPrice"].(float64); ok {
        return usdPrice, nil
    }

    // Log if usdPrice is missing
    log.Printf("usdPrice not found in Moralis API response for token: %s", tokenAddress)
    return 0, fmt.Errorf("usdPrice not found in Moralis API response for token: %s", tokenAddress)
}

func CalculateTokensSold(supply float64, holders []TokenHolder) float64 {
    totalHeld := 0.0
    for _, holder := range holders {
        totalHeld += holder.Balance
    }
    return supply - totalHeld
}