package main

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"

	"github.com/gorilla/websocket"
)

// PairNotification represents the structure of new pair notifications.
type PairNotification struct {
	Params struct {
		Slot int64 `json:"slot"`
		Pair struct {
			SourceExchange          string `json:"sourceExchange"`
			AmmAccount              string `json:"ammAccount"`
			BaseTokenLiquidityAdded string `json:"baseTokenLiquidityAdded"`
			QuoteTokenLiquidityAdded string `json:"quoteTokenLiquidityAdded"`
			BaseToken               struct {
				Account string `json:"account"`
				Info    struct {
					Decimals int    `json:"decimals"`
					Supply   string `json:"supply"`
					Metadata struct {
						Name   string `json:"name"`
						Symbol string `json:"symbol"`
						Logo   string `json:"logo"`
					} `json:"metadata"`
				} `json:"info"`
			} `json:"baseToken"`
			QuoteToken struct {
				Account string `json:"account"`
			} `json:"quoteToken"`
		} `json:"pair"`
	} `json:"params"`
}

// connectWebSocket establishes a WebSocket connection.
func connectWebSocket() *websocket.Conn {
	headers := http.Header{
		"X-Api-Key": []string{"be39af20a1388a7a9bf3368f0b5e841c"},
	}
	conn, _, err := websocket.DefaultDialer.Dial("wss://api.solanastreaming.com/", headers)
	if err != nil {
		log.Fatal("Error connecting to WebSocket:", err)
	}
	log.Println("Connected to WebSocket.")
	return conn
}

// SubscribeToNewPairs sends a subscription request to the WebSocket server.
func SubscribeToNewPairs(conn *websocket.Conn) {
	subscriptionMessage := `{"id":1,"method":"newPairSubscribe"}`
	err := conn.WriteMessage(websocket.TextMessage, []byte(subscriptionMessage))
	if err != nil {
		log.Fatal("Error subscribing to new pairs:", err)
	}
	log.Println("Subscription request sent.")
}

// ParsePairNotification parses incoming WebSocket messages for pair notifications.
func ParsePairNotification(message []byte) (*PairNotification, string) {
	var response map[string]interface{}
	if err := json.Unmarshal(message, &response); err != nil {
		log.Println("Error parsing message:", err)
		return nil, ""
	}

	log.Printf("Unmarshaled JSON: %+v", response)

	if response["result"] != nil {
		return nil, "subscription"
	}

	if params, ok := response["params"].(map[string]interface{}); ok {
		var notification PairNotification
		paramsBytes, _ := json.Marshal(params)
		if err := json.Unmarshal(paramsBytes, &notification.Params); err != nil {
			log.Println("Error parsing PairNotification:", err)
			return nil, ""
		}
		return &notification, "newPairNotification"
	}

	return nil, "unknown"
}

// Main application logic
func main() {
	conn := connectWebSocket()
	defer conn.Close()

	SubscribeToNewPairs(conn)

	bot := SetupTelegramBot()

	for {
		_, message, err := conn.ReadMessage()
		if err != nil {
			log.Println("Error reading WebSocket message:", err)
			break
		}

		log.Printf("Raw WebSocket message: %s", message)

		// Parse the WebSocket message
		pairNotification, messageType := ParsePairNotification(message)

		switch messageType {
		case "subscription":
			log.Println("Subscription confirmed, waiting for new pair notifications...")
		case "newPairNotification":
			// Process the new pair notification
			pair := pairNotification.Params.Pair

			// Fetch token supply and largest accounts
			supply, decimals, err := FetchTokenSupply(pair.BaseToken.Account)
			if err != nil {
				log.Printf("Error fetching token supply: %v", err)
				continue
			}

			holders, err := FetchLargestTokenAccounts(pair.BaseToken.Account)
			if err != nil {
				log.Printf("Error fetching largest token accounts: %v", err)
				continue
			}

            // Fetch token price with retry mechanism
            price,err := FetchTokenPriceFromMoralis(pair.BaseToken.Account)
            // Format market cap
            marketCap := FormatMarketCap(supply, decimals, price)
    
			topHolders := ""
			for _, holder := range holders {
				topHolders += fmt.Sprintf("Address: %s, Balance: %.2f\n", holder.Address, holder.Balance)
			}

			bundledWallets := AnalyzeWallets(holders)
			rugPullRisk := AssessRugPullRisk(holders)

			// Send the results to Telegram
			


            // Compute market cap
            marketCap = FormatMarketCap(supply, decimals, price)
			tokensSold := CalculateTokensSold(supply, holders)


            // Notify Telegram with the correct price and market cap
            NotifyTelegram(
                bot,
                marketCap+" USD",
                fmt.Sprintf("%.6f USD", price), // Use the fetched price
                bundledWallets,
                topHolders,
                rugPullRisk,
                pair.BaseToken.Info.Metadata.Name,
                pair.BaseToken.Info.Metadata.Symbol,
                pair.BaseToken.Account,
            	fmt.Sprintf("%.6f token sold", tokensSold),
            )
    
        default:
			log.Println("Unknown message type received, ignoring...")
		}
	}
}
